/**
  *******************************************************************************
  * 
  * @file    hal_t2n.c
  * @brief   ES7P203 t2n module driver.
  *
  * @version v1.0
  * @date 22 Aug 2018
  * @author  AE Team
  * @note
  *
  * copyright (C) shanghai Eastsoft Mictroelectornics Co. Ltd. All rights reseverd.
  *
  @verbatim
  =================================================================================
  			##### How to use this driver #####
  =================================================================================
  [..]  The T2n driver can be used as follows:
    (+) Initialize the t2n registers by using t2n_init().
    (+) Start t2n timer by using t2n_start()
    (+) Get timer value by using t2n_get_value()

  @endverbatim
  *********************************************************************************
  */
#include "hal_t2n.h"

/**
  * @addtogroup ES7P203_HAL
  * @{
  */

/**
  * @defgroup t2n base
  * @brief t2n module driver
  * @{
  */

/**
  * @defgroup T2n_Public_Functions t2n Public Functions
  * @brief    T2n Functions
@verbatim
=================================================================================
		##### Flash Public Functions #####
=================================================================================
  [..]  The t2n driver can be used as follows:
    (+) Initialize the t2n registers by using t2n_init().
    (+) Start t2n timer by using t2n_start()
    (+) Get timer value by using t2n_get_value()
@endverbatim
  * @{
  */

/**
  * @brief  Init t2n
  * @param  type: The type of t2n
  * @param  init: The point of init struct
  * @retval Status, see @ref hal_status_t.
  */
hal_status_t t2n_init(t2n_type_t type, t2n_init_t *init)
{
	if (init == NULL)
		return ERROR;
	
	assert_param(IS_T2N_TYPE(type));
	assert_param(IS_T2N_MODE_TYPE(init->mode));
	assert_param(IS_T2N_POLARITY_TYPE(init->polarity));
	assert_param(IS_T2N_CLOCK_DIV_TYPE(init->clk_div));
	assert_param(IS_T2N_PWMN_TYPE(init->pwm_en));
	assert_param(IS_T2N_BEHIND_FREQ_TYPE(init->pos));

	switch (type) {
	case T20_TYPE:
		T20EN = 0;
		MODIFY_REG(T20CM, 0xF, init->clk_div);
		MODIFY_REG(T20CH, 0x7F, (init->pos - 1));
		MODIFY_REG(T20CL, (0xF << 4), (init->mode << 4));
		T20PH = (init->period >> 8);
		T20PL = init->period;
		if (init->mode == T2N_DOUBLE_PWM_MODE) {
			T20R0L = init->pwm0_prec;
			T20R0H = (init->pwm0_prec >> 8);
			T20R1L = init->pwm1_prec;
			T20R1H = (init->pwm1_prec >> 8);
			MODIFY_REG(T20CM, (0x3 << 4), (init->polarity << 4));
			switch (init->pwm_en) {
			case PWM0_EN:
				P200EN = 1;
				break;
			case PWM1_EN:
				P201EN = 1;
				break;
			case PWM_ALL_EN:
				P200EN = 1;
				P201EN = 1;
				break;
			default:
				break;
			}
		}
		break;
	case T21_TYPE:
		T21EN = 0;
		MODIFY_REG(T21CM, 0xF, init->clk_div);
		MODIFY_REG(T21CH, 0x7F, (init->pos - 1));
		MODIFY_REG(T21CL, (0xF << 4), (init->mode << 4));
		T21PH = (init->period >> 8);
		T21PL = init->period;
		if (init->mode == T2N_DOUBLE_PWM_MODE) {
			T21R0L = init->pwm0_prec;
			T21R0H = (init->pwm0_prec >> 8);
			T21R1L = init->pwm1_prec;
			T21R1H = (init->pwm1_prec >> 8);
			MODIFY_REG(T21CM, (0x3 << 4), (init->polarity << 4));
			switch (init->pwm_en) {
			case PWM0_EN:
				P210EN = 1;
				break;
			case PWM1_EN:
				P211EN = 1;
				break;
			case PWM_ALL_EN:
				P210EN = 1;
				P211EN = 1;
				break;
			default:
				break;
			}
		}
		break;
	default:
		break;
	}
	
	return OK;
}

/**
  * @brief  Start t2n
  * @param  type: The type of t2n
  * @retval None.
  */
void t2n_start(t2n_type_t type)
{
	assert_param(IS_T2N_TYPE(type));
	
	switch (type) {
	case T20_TYPE:
		if ((T20CL & 0xc0) == 0xc0)
			T20TR = 1;

		T20EN = 1;
		return;
	case T21_TYPE:
		if ((T21CL & 0xc0) == 0xc0)
			T21TR = 1;

		T21EN = 1;
		return;
	default:
		return;
	}
}

/**
  * @brief  Get t2n value
  * @param  type: The type of t2n
  * @retval value.
  */
uint16_t t2n_get_value(t2n_type_t type)
{
	assert_param(IS_T2N_TYPE(type));
	switch (type) {
	case T20_TYPE:
		return ((T20H << 8) | T20L);
	case T21_TYPE:

		return ((T21H << 8) | T21L);
	default:
		return 0;
	}
}

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */
